/**
 * This is example code used in Front-Commerce demos to configure analytics plugins.
 * It provides configuration for Google Analytics through environment variables:
 * - GOOGLE_ANALYTICS_ENABLED: to enable/disable Google Analytics
 * - GOOGLE_ANALYTICS_MEASUREMENT_IDS: comma-separated list of GA4 measurement IDs
 *
 * Projects can:
 * - reuse this configuration provider as-is
 * - adapt it to their needs by copying and modifying it
 * - or create their own configuration provider from scratch
 *
 * See https://developers.front-commerce.com/docs/advanced/analytics/plugins
 * for more details about analytics configuration in Front-Commerce.
 */

const analyticsConfigProvider = () => {
  const memoCache = new WeakMap();
  return {
    name: "analyticsConfiguration",
    schema: () => {
      return {
        public: {
          analytics: {
            googleAnalytics: {
              enabled: {
                doc: "Enable Google Analytics",
                format: Boolean,
                default: false,
                env: "GOOGLE_ANALYTICS_ENABLED",
              },
              measurementIds: {
                doc: "The measurement IDs to use for Google Analytics",
                format: "comma-separated-string",
                default: [],
                env: "GOOGLE_ANALYTICS_MEASUREMENT_IDS",
              },
            },
          },
        },
      };
    },

    // Since we want to extend existing config providers, we
    // do this in slowValuesOnEachRequest, and memoize the
    // static values.
    slowValuesOnEachRequest: (req: any, config: any) => {
      if (memoCache.has(config)) {
        return memoCache.get(config);
      }

      const csp = (config.contentSecurityPolicy || {}) as any;
      const result = {
        contentSecurityPolicy: {
          ...csp,
          directives: {
            ...csp.directives,
            scriptSrc: [
              ...csp.directives.scriptSrc,
              "https://www.googletagmanager.com",
            ],

            imgSrc: [
              ...csp.directives.imgSrc,
              "*.google-analytics.com",
              "*.analytics.google.com",
            ],

            connectSrc: [
              ...csp.directives.connectSrc,
              "*.google-analytics.com",
              "*.analytics.google.com",
            ],
          },
        },
      };

      memoCache.set(config, result);
      return result;
    },
  };
};
export default analyticsConfigProvider;
