/**
 * By default, Remix will handle generating the HTTP Response for you.
 * You are free to delete this file if you'd like to, but if you ever want it revealed again, you can run `npx remix reveal` ✨
 * For more information, see https://remix.run/docs/en/main/file-conventions/entry.server
 */

import { PassThrough } from "node:stream";
import type { AppLoadContext, EntryContext } from "@remix-run/node";
import { RemixServer } from "@remix-run/react";
import { isbot } from "isbot";
import { renderToPipeableStream } from "react-dom/server";
import { FrontCommerceServer } from "@front-commerce/remix/react";
import prepareV2ServerRenderedApp from "@front-commerce/compat/apollo/prepareV2ServerRenderedApp";
import manifest from "virtual:front-commerce/manifest";

const ABORT_DELAY = 20_000; // prevent an error "The render was aborted by the server without a reason" for slow responses

export default async function handleRequest(
  request: Request,
  responseStatusCode: number,
  responseHeaders: Headers,
  remixContext: EntryContext,
  loadContext: AppLoadContext,
) {
  return isbot(request.headers.get("user-agent"))
    ? handleBotRequest(
        request,
        responseStatusCode,
        responseHeaders,
        remixContext,
        loadContext,
      )
    : handleBrowserRequest(
        request,
        responseStatusCode,
        responseHeaders,
        remixContext,
        loadContext,
      );
}

function handleBotRequest(
  request: Request,
  responseStatusCode: number,
  responseHeaders: Headers,
  remixContext: EntryContext,
  loadContext: AppLoadContext,
) {
  return new Promise(async (resolve, reject) => {
    const { pipe, abort } = renderToPipeableStream(
      await prepareV2ServerRenderedApp(
        <FrontCommerceServer
          context={loadContext.frontCommerce}
          remixContext={remixContext}
        >
          <RemixServer
            context={remixContext}
            url={request.url}
            abortDelay={ABORT_DELAY}
          />
        </FrontCommerceServer>,

        manifest,
        request,
        loadContext.frontCommerce.config,
      ),
      {
        onAllReady() {
          const body = new PassThrough();

          responseHeaders.set("Content-Type", "text/html");

          resolve(
            new Response(body, {
              headers: responseHeaders,
              status: responseStatusCode,
            }),
          );

          pipe(body);
        },
        onShellError(error: unknown) {
          reject(error);
        },
        onError(error: unknown) {
          responseStatusCode = 500;
          console.error(error);
        },
      },
    );

    setTimeout(abort, ABORT_DELAY);
  });
}

function handleBrowserRequest(
  request: Request,
  responseStatusCode: number,
  responseHeaders: Headers,
  remixContext: EntryContext,
  loadContext: AppLoadContext,
) {
  loadContext.frontCommerce.services.ServerTimings.start("SSR");
  return new Promise(async (resolve, reject) => {
    const { pipe, abort } = renderToPipeableStream(
      await prepareV2ServerRenderedApp(
        <FrontCommerceServer
          context={loadContext.frontCommerce}
          remixContext={remixContext}
        >
          <RemixServer
            context={remixContext}
            url={request.url}
            abortDelay={ABORT_DELAY}
          />
        </FrontCommerceServer>,
        manifest,
        request,
        loadContext.frontCommerce.config,
      ),
      {
        onShellReady() {
          const body = new PassThrough();

          responseHeaders.set("Content-Type", "text/html");

          loadContext.frontCommerce.services.ServerTimings.end("SSR");
          resolve(
            new Response(body, {
              headers: responseHeaders,
              status: responseStatusCode,
            }),
          );

          pipe(body);
        },
        onShellError(error: unknown) {
          loadContext.frontCommerce.services.ServerTimings.end("SSR");
          reject(error);
        },
        onError(error: unknown) {
          console.error(error);
          responseStatusCode = 500;
        },
      },
    );

    setTimeout(abort, ABORT_DELAY);
  });
}
