// This is an example route, to learn more see https://developers.front-commerce.com/docs/3.x/get-started/your-first-route

import { FrontCommerceApp } from "@front-commerce/remix";
import { json } from "@front-commerce/remix/node";
import { useLoaderData } from "@front-commerce/remix/react";
import type { LoaderFunctionArgs, MetaFunction } from "@remix-run/node";
import { defineMessages } from "react-intl";

import ButtonLink from "theme/components/atoms/Typography/Link/ButtonLink";

const messages = defineMessages({
  title: {
    defaultMessage: "Hello World",
    id: "hello-world.title",
  },
});

type Response = {
  title: string;
  shop: {
    url: string;
  };
  me: {
    firstname: string;
  };
  navigationMenu: {
    id: string;
    name: string;
    path: string;
  }[];
};

export const loader = async ({ context }: LoaderFunctionArgs) => {
  const app = new FrontCommerceApp(context.frontCommerce);

  await app.services.RateLimiter.limitHTTPResource(
    "/hello-world",
    app.user.clientIp,
    {
      max: 5,
      duration: "1m",
    },
  );

  const response = await app.graphql.query(/* GraphQL */ `
    {
      shop {
        url
      }
      me {
        firstname
      }
      navigationMenu {
        id
        name
        path
      }
    }
  `);

  const data = {
    ...(response as any),
    title: app.intl.formatMessage(messages.title),
  } as Response;

  return json(data);
};

export const meta: MetaFunction<typeof loader> = (args) => {
  return [{ title: args.data.title }];
};

export default function Pokemon() {
  const data = useLoaderData<typeof loader>();

  return (
    <div
      style={{
        display: "flex",
        justifyContent: "center",
        alignItems: "center",
        minHeight: "100vh",
      }}
    >
      <div style={{ textAlign: "center" }}>
        <h1>Hi {data.me?.firstname ?? "visitor"} 👋</h1>
        <h3>Welcome to {data.shop.url}</h3>
        <h4>Here are the main categories</h4>
        <div
          style={{
            margin: "auto",
            display: "grid",
            gridTemplateColumns: "repeat(3, 1fr)",
            gap: "16px",
          }}
        >
          {data.navigationMenu.map(({ id, name, path }) => (
            <ButtonLink key={id} buttonAppearance="default" to={path}>
              {name}
            </ButtonLink>
          ))}
        </div>
      </div>
    </div>
  );
}
