import {
  BlocksContentType,
  StorefrontContentType,
} from "@front-commerce/contentful";
import gql from "graphql-tag";

/**
 * @param {BlocksContentType} homePageBlocks
 * @param {InstanceType<import("./SeoMetadata")["default"]>} seoMetadata
 * @param {InstanceType<import("./Page")["default"]>} page
 */
const formatContentfulData =
  (homePageBlocks, seoMetadata, page) => (contentfulData) => {
    return {
      title: contentfulData.title,
      seo: seoMetadata.dataFormatter(contentfulData.seo),
      blocks: homePageBlocks.dataFormatter(
        contentfulData.contentBlocksCollection?.items ?? [],
      ),
      articles: (contentfulData.articlesCollection?.items ?? []).map(
        (article) => page.dataFormatter(article),
      ),
    };
  };

class HomePage extends StorefrontContentType {
  /**
   * @param {import("./Blocks")["default"]} blocksLibrary
   * @param {InstanceType<import("./SeoMetadata")["default"]>} seoMetadata
   * @param {InstanceType<import("./Page")["default"]>} page
   */
  constructor(blocksLibrary, seoMetadata, page) {
    const homePageBlocks = new BlocksContentType(
      "home",
      "HomeContentBlocksItem",
      [blocksLibrary.Carousel, blocksLibrary.CategoryProductsList],
      "HomePageBlocks",
    );

    super(
      "home",
      "HomePage",
      formatContentfulData(homePageBlocks, seoMetadata, page),
      "homePage",
    );
    this.homePageBlocks = homePageBlocks;
    this.seoMetadata = seoMetadata;
    this.page = page;
  }

  get contentfulFragment() {
    return gql`
      fragment HomeFragment on Home {
        ${this.sys}
        title
        seo {
          ${this.seoMetadata.contentfulFragmentName}
        }
        contentBlocksCollection {
          items {
            __typename
            ${this.homePageBlocks.contentfulFragmentName}
          }
        }
        articlesCollection {
          items{
            __typename
            ${this.page.contentfulFragmentName}
          }
        }
      }
      ${this.seoMetadata.contentfulFragment}
      ${this.homePageBlocks.contentfulFragment}
      ${this.page.contentfulFragment}
    `;
  }
}

export default HomePage;
