import { createGraphQLRuntime } from "@front-commerce/core/graphql";
import blocksLibrary from "./contentType/Blocks";
import HomePage from "./contentType/HomePage";
import Page from "./contentType/Page";
import RichTextField from "./contentType/RichTextField";
import SeoMetadata from "./contentType/SeoMetadata";
import MarketingCmsLoader from "./MarketingCmsLoader";

export default createGraphQLRuntime({
  resolvers: {
    Query: {
      homePage: async (_, _args, { loaders }) => {
        return loaders.MarketingCms.loadHomePage();
      },
      contactPage: async (_, _args, { loaders }) => {
        return loaders.MarketingCms.loadContactPage();
      },
      marketingPage: async (_, { slug }, { loaders }) => {
        return loaders.MarketingCms.loadPageBySlug(slug);
      },
      allMarketingPages: async (_, _args, { loaders }) => {
        return loaders.MarketingCms.loadAllPages();
      },
    },
    WysiwygContentfulLinkData: {
      url: async (data, _, { loaders }) => {
        try {
          const entryId = data.node.attrs.find(
            ({ name }: any) => name === "href",
          )?.value;

          const url = await loaders.Contentful.loadLocalRoute("Entry", entryId);

          return url;
        } catch (e) {
          return "#TODO";
        }
      },
    },
  },

  contextEnhancer: ({ loaders }) => {
    // Note: this should be either a slug or an app configuration in a real project
    const contactPageId = "dntmgP2QV6uRuRChrdLFj";

    /**
     * @type {InstanceType<import("front-commerce-contentful/server/core/ContentfulLoader")["default"]>}
     */
    const ContentfulLoader = loaders.Contentful;

    const seoMetadata = new SeoMetadata();
    const richTextField = new RichTextField(loaders.Wysiwyg);

    const page = new Page(seoMetadata, richTextField);

    const AppContentTypes = {
      page,
      home: new HomePage(blocksLibrary, seoMetadata, page),
    };

    const MarketingCms = new MarketingCmsLoader(
      ContentfulLoader,
      AppContentTypes,
      {
        contactPageId,
      },
    );

    for (const contentType of Object.values(AppContentTypes)) {
      ContentfulLoader.registerStorefrontContentType(contentType);
    }

    for (const contentType of Object.values(blocksLibrary)) {
      ContentfulLoader.registerStorefrontContentType(contentType);
    }

    ContentfulLoader.routes.registerUrlMatcher("Page", async (id: string) => {
      // Example of special case handling, to enforce a URL
      // for a page as a static value.
      if (MarketingCms.isContactPage(id)) {
        return `/contact`;
      }

      const page = await MarketingCms.loadPageById(id);
      if (page) {
        return `/content/${page.slug}`;
      }
      return null;
    });

    return {
      MarketingCms,
    };
  },
});
