import { defer } from "@front-commerce/remix/node";
import { useLoaderData } from "@front-commerce/remix/react";
import type { LoaderFunction } from "@remix-run/node";
import { Await } from "@remix-run/react";
import { Suspense, useEffect, useState } from "react";

export const headers = () => ({
  "Cache-Control": "public, s-maxage=10, stale-while-revalidate=50",
});

const nowInHHMMSS = () => {
  const now = new Date();
  return `${now.getHours()}:${now.getMinutes()}:${now.getSeconds()}`;
};

export const loader: LoaderFunction = async () => {
  const createDelayedPromise = (delay: number) => {
    return new Promise((resolve) =>
      setTimeout(
        () =>
          resolve(
            `This is delayed after ${delay}ms / server date -> ${nowInHHMMSS()}`,
          ),
        delay,
      ),
    );
  };

  const blocking200 = await createDelayedPromise(200);
  const delayed1s = createDelayedPromise(1000);
  const delayed5s = createDelayedPromise(5000);
  const delayed10s = createDelayedPromise(10000);
  const unused12s = createDelayedPromise(12000);

  return defer({
    immediate: `This is not delayed: server date -> ${nowInHHMMSS()}`,
    blocking200,
    delayed1s,
    delayed5s,
    delayed10s,
    unused12s,
  });
};

const Clock = () => {
  const [date, setDate] = useState("");

  useEffect(() => {
    const interval = setInterval(() => setDate(nowInHHMMSS()), 1000);
    return () => clearInterval(interval);
  }, []);

  return <p>Current date: {date}</p>;
};

const ClientDate = () => {
  const [date, setDate] = useState("");
  useEffect(() => {
    setDate(nowInHHMMSS());
  }, []);
  return <em>Client date: {date}</em>;
};

export default function DeferExample() {
  const data = useLoaderData<typeof loader>();

  return (
    <div className="container">
      <h1>Defer Example</h1>
      <p>
        This route is rendered on the server, with data artificially delayed
        server side and progressively displayed.
      </p>

      <Clock />

      <div className="deferred-block">
        <h2>Immediate data</h2>
        <p>
          {data.immediate} / <ClientDate />
        </p>
      </div>

      <div className="deferred-block">
        <h2>Blocking: 200ms</h2>
        <p>
          {data.blocking200} / <ClientDate />
        </p>
      </div>

      <div className="deferred-block">
        <h2>Delayed: 1 second</h2>
        <Suspense fallback="Loading...">
          <Await resolve={data.delayed1s}>
            {(delayed1s) => (
              <p>
                {delayed1s} / <ClientDate />
              </p>
            )}
          </Await>
        </Suspense>
      </div>

      <div className="deferred-block">
        <h2>Delayed: 5 seconds</h2>
        <Suspense fallback="Loading...">
          <Await resolve={data.delayed5s}>
            {(delayed5s) => (
              <p>
                {delayed5s} / <ClientDate />
              </p>
            )}
          </Await>
        </Suspense>
      </div>

      <div className="deferred-block">
        <h2>Delayed: 10 seconds</h2>
        <p>
          <em>
            Requires <code>const ABORT_DELAY = 20_000;</code> (greater than 10s)
            in <code>entry.server.tsx</code>
          </em>
        </p>
        <Suspense fallback="Loading...">
          <Await resolve={data.delayed10s}>
            {(delayed10s) => (
              <p>
                {delayed10s} / <ClientDate />
              </p>
            )}
          </Await>
        </Suspense>
      </div>
    </div>
  );
}
