import { json } from "@front-commerce/remix/node";
import { useLoaderData } from "@front-commerce/remix/react";
import { render, screen, fireEvent, act } from "@testing-library/react";
import { createRemixStub } from "@remix-run/testing";
import FaqEntry from "theme/modules/Faq/FaqEntry";
import { IntlProvider } from "react-intl";
import type { FaqEntryQueryQuery } from "~/graphql/graphql";

describe("FaqEntry Page", () => {
  // Mocking the loader response data
  const mockLoaderData = {
    faqEntry: {
      slug: "test-slug",
      question: "What is test?",
      answer: "Test is a test.",
    } satisfies FaqEntryQueryQuery["faqEntry"],
  };

  // Component that uses useLoaderData to receive the mocked loader data
  const FaqEntryLayout = () => {
    const data = useLoaderData<typeof mockLoaderData>();
    return (
      <IntlProvider locale="en" messages={{}}>
        <FaqEntry entry={data.faqEntry} />
      </IntlProvider>
    );
  };

  // Creating a RemixStub for testing
  const RemixStub = createRemixStub([
    {
      path: "/",
      Component: FaqEntryLayout,
      loader() {
        return json(mockLoaderData);
      },
      async action({ request }) {
        const formData = await request.formData();

        const success = formData.get("isFaqUseful") === "true";

        return {
          success,
        };
      },
    },
  ]);

  it("should render the Faq Entry", async () => {
    render(<RemixStub />);

    await screen.findByText("Get permalink for What is test?");
    await screen.findByText("Test is a test.");
  });

  it("should return with a success true result", async () => {
    render(<RemixStub />);

    await screen.findByText("Get permalink for What is test?");
    await screen.findByText("Test is a test.");

    const button = await screen.findByTitle("Oui");

    fireEvent.click(button);

    await screen.findByText("Sending...");

    await screen.findByText("Your feedback was sent successfully.");
  });

  it("should return with a success false result", async () => {
    render(<RemixStub />);

    await screen.findByText("Get permalink for What is test?");
    await screen.findByText("Test is a test.");

    const button = await screen.findByTitle("No");

    act(() => {
      fireEvent.click(button);
    });

    await screen.findByText("Sending...");

    await screen.findByText("An error occurred when sending your feedback.");
  });
});
