import { FrontCommerceApp } from "@front-commerce/remix";
import { generateMetas, json } from "@front-commerce/remix/node";
import {
  useLoaderData,
  type UseDataFunctionReturn,
} from "@front-commerce/remix/react";
import type {
  ActionFunctionArgs,
  LoaderFunctionArgs,
  MetaArgs,
} from "@remix-run/node";
import { isRouteErrorResponse, useRouteError } from "@remix-run/react";
import {
  FaqEntryQueryDocument,
  VoteIsFaqUsefulDocument,
} from "~/graphql/graphql";
import FaqEntry from "theme/modules/Faq/FaqEntry";
import QuestionVotedEvent from "../server-events/events/QuestionVotedEvent";
import { createHandle } from "@front-commerce/remix/handle";

// The route exports a `loader` function that is responsible
// for fetching data and throwing errors, ensuring that
// the main component is only rendered in the "happy path".
export const loader = async ({ context, params }: LoaderFunctionArgs) => {
  if (!params.slug) {
    throw json("Page not found", {
      status: 404,
    });
  }

  // The `loader` uses the `FrontCommerceApp` class to access the Front-Commerce
  // core from Remix. In this example, we use it to fetch data from the GraphQL
  // unified API (similar to the one you're used to).
  const app = new FrontCommerceApp(context.frontCommerce);
  const response = await app.graphql.query(FaqEntryQueryDocument, {
    input: { slug: params.slug },
  });
  if (!response.faqEntry) {
    throw new Response("Question not found", { status: 404 });
  }

  return json({
    faqEntry: response.faqEntry,
  });
};

export const action = async ({
  context,
  params,
  request,
}: ActionFunctionArgs) => {
  const { slug } = params;

  const formData = await request.formData();

  const isFaqUseful = formData.get("isFaqUseful") === "true";
  // The `action` uses the `FrontCommerceApp` class to access the Front-Commerce
  // core from Remix. In this example, we use it to mutate data in the GraphQL
  // unified API (similar to the one you're used to).
  const app = new FrontCommerceApp(context.frontCommerce);
  const response = await app.graphql.mutate(VoteIsFaqUsefulDocument, {
    input: {
      slug,
      isFaqUseful,
    },
  });

  // We call the `ServerEventBus` to dispatch an event.
  // In this example, we use it to dispatch a `QuestionVotedEvent` event.
  // We will instanciate a `QuestionVotedEvent` class with the data we want to
  // collect for this event.
  app.services.ServerEventBus.publish(
    new QuestionVotedEvent(String(slug), isFaqUseful, app.config.currentShopId),
  );

  if (!response.voteIsFaqUseful) {
    throw new Response("Question not found", { status: 404 });
  }

  return json({
    success: response.voteIsFaqUseful?.success ?? false,
  });
};

export const meta = (args: MetaArgs) => {
  const data = args.data as UseDataFunctionReturn<typeof loader>;

  return generateMetas((metas) => {
    // Here we show how you can attached the layout route meta to the
    // current route meta, this will result in the following title:
    // "Frequently Asked Questions • How to use the product?"
    const titleParts = metas.matches
      .filter((match) => match.pathname.startsWith("/faq"))
      .map((match) => {
        return match.meta.find((m) => "title" in m);
      })
      .filter(Boolean)
      .reduce((acc, meta) => {
        if ("title" in meta) {
          acc += `${meta.title} •`;
        }
        return acc;
      }, "");

    return [{ title: `${titleParts} ${data?.faqEntry?.question}` }];
  }, `/faq/${data?.faqEntry?.slug}`)(args);
};

export const handle = createHandle({
  sitemapFetcher: "faq",
});

// The main component is a plain React component that receives
// the data from the loader, using Remix fetching primitives (`useLoaderData`)
// both on the server and on the client.
export default function Component() {
  const { faqEntry } = useLoaderData<typeof loader>();

  return <FaqEntry entry={faqEntry} />;
}

// The route also exports an ErrorBoundary component that is responsible
// for displaying errors. It can be used to display a custom error page.
export const ErrorBoundary = () => {
  const error = useRouteError();

  if (isRouteErrorResponse(error)) {
    return <div>FAQ : question not found</div>;
  }
};
