import { json } from "@front-commerce/remix/node";
import { useFetcher, useLoaderData } from "@front-commerce/remix/react";
import { SubmitButton } from "theme/components/atoms/Button";
import Stack from "theme/components/atoms/Layout/Stack";
import { mockApi, type Services } from "../mockApi";
import { useEffect, useState } from "react";
import { useNavigate } from "@remix-run/react";
import type { action } from "./[__front-commerce].demo-services";

export const loader = async () => {
  const services = await mockApi.readServices();

  return json({ services });
};

const ServiceButton = (props: { name: keyof Services }) => {
  const [loading, setLoading] = useState(false);
  const fetcher = useFetcher<typeof action>();
  const navigate = useNavigate();

  useEffect(() => {
    if (fetcher.state === "idle") return;

    setLoading(true);
  }, [fetcher.state]);

  useEffect(() => {
    if (fetcher.data?.success) {
      setTimeout(() => {
        setLoading(false);
        navigate("/maintenance");
      }, 5000 + 1000); // We memoize the maintenance status for 5 seconds, so we need to wait for that
    }
  }, [fetcher.data, navigate]);

  return (
    <fetcher.Form method="post" action={`/__front-commerce/demo-services`}>
      <input type="hidden" name="name" value={props.name} />
      <SubmitButton
        state={loading ? "pending" : "default"}
        id={`trigger-service-down-${props.name}`}
      >
        {props.name}
      </SubmitButton>
    </fetcher.Form>
  );
};

export default function MaintenanceDemoPage() {
  const { services } = useLoaderData<typeof loader>();
  const servicesNames = Object.keys(services) as (keyof Services)[];

  return (
    <div className="container">
      <Stack>
        <h1>Demo Services</h1>
        <h4 style={{ maxWidth: 600, width: "100%" }}>
          These are services which have been added to the maintenance mode
          health checks, if one of these services return `false` the page will
          enter in maintenance mode
        </h4>
        <hr />
        <Stack>
          {servicesNames.map((service) => (
            <ServiceButton key={service} name={service as any} />
          ))}
        </Stack>
      </Stack>
    </div>
  );
}
