import { createGraphQLRuntime } from "@front-commerce/core/graphql";
import { PrismicDemoCoreLoader } from "./loaders";
import { withSliceFields } from "./SliceExtractor";
import type { IntegrationFieldResolverContent } from "@front-commerce/prismic/graphql";
import type {
  CategoryIntegrationField,
  ProductIntegrationField,
} from "../../integration-fields";

export default createGraphQLRuntime({
  contextEnhancer: ({ config, loaders }) => {
    const domains = [
      config.shop.url,
      /.*\.front-commerce.app/,
      /.*\.platform.site/,
      /^http:\/\/localhost/,
    ];

    return {
      PrismicDemoCore: PrismicDemoCoreLoader(
        loaders.Prismic,
        loaders.Wysiwyg,
        domains,
      ),
    };
  },

  resolvers: {
    FeaturedProductsList: withSliceFields({
      category: async (
        content: IntegrationFieldResolverContent<
          "category",
          typeof CategoryIntegrationField
        >,
      ) => {
        return content.category.loadValue().catch(() => null);
      },
    }),
    ProductsList: withSliceFields({
      products: async (content) => {
        return Promise.all(
          content.items.map(
            (
              content: IntegrationFieldResolverContent<
                "product",
                typeof ProductIntegrationField
              >,
            ) => {
              return content.product.loadValue().catch(() => null);
            },
          ),
        ).then((products = []) => products.filter(Boolean));
      },
    }),
    Carousel: withSliceFields({
      slides: (content) => {
        return content.items;
      },
    }),
    CarouselSlide: {
      title: (content) => content.slide_title,
      image: (content) => content.slide_image,
      cta: (content) => ({
        url: content.slide_cta_url,
        text: content.slide_cta_text,
      }),
    },
    Push: withSliceFields({
      blocks: (content) => content.items,
    }),
    PushBlock: {
      title: (content) => {
        return content.push_title;
      },
      image: (content) => content.push_block_image,
      cta: (content) => ({
        url: content.push_url,
        text: content.push_link_text,
      }),
      format: (content) => content.push_block_format,
      cellSize: (content) => content.cell_size,
    },
    RichText: withSliceFields({
      content: ({ content }) => content,
    }),
    CustomEmbed: withSliceFields({
      html: (content) => content.html,
    }),
  },
});
